### Summary ####
# Input: No dataset, but data from the EIP Dashboard (Bureau of the Fiscal Service)
# Output: Figure II and III in Section I The Economic Impact Payments
# Outline:
# 1. Figure II
# 2. Figure III

### Preparations ####
library(dplyr) # For data processing
library(ggplot2) # For plots
library(lessR) # For piecharts
library(egg)
library(grid)
library(gridExtra)

### Figure II ####

# Data Source: Department of Treasury

# First EIP 

# Single with no qualifying child
sin_with_no_k <- function(x) ifelse(0 <= x & x <= 75000, 1200,
                                    ifelse(x >= 75000 & x <= 99000,  1200-0.05*(x-75000),
                                           ifelse(x >= 99000, 0, 1200)))

# Head of household with one qualifying child
hoh_with_one_k <- function(x) ifelse(0 <= x & x <= 112500, 1700,
                                     ifelse(x >= 112500 & x <= 146500, 1700-0.05*(x-112500),
                                            ifelse(x >= 146500, 0, 1700)))

# Married Filing Jointly with no qualifying child
join_with_no_k <- function(x) ifelse(0 <= x & x <= 150000, 2400,
                                     ifelse(x >= 150000 & x <= 198000,  2400-0.05*(x-150000),
                                            ifelse(x >= 198000, 0, 2400)))

# Married Filing Jointly with one qualifying child
join_with_one_k <- function(x) ifelse(0 <= x & x <= 150000, 2900,
                                      ifelse(x >= 150000 & x <= 208000,  2900-0.05*(x-150000),
                                             ifelse(x >= 208000, 0, 2900)))

# Married Filing Jointly with two qualifying child
join_with_two_k <- function(x) ifelse(0 <= x & x <= 150000, 3400,
                                      ifelse(x >= 150000 & x <= 218000,  3400-0.05*(x-150000),
                                             ifelse(x >= 218000, 0, 3400)))

p <- ggplot(data = data.frame(x = 0), mapping = aes(x = x))

p + geom_smooth(stat="function",
                fun = sin_with_no_k,
                aes(color = "Single without a qualifying child" ,
                    linetype = "Single without a qualifying child"),
                se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = hoh_with_one_k,
              aes(color = "Head of household with one qualifying child",
                  linetype= "Head of household with one qualifying child"),
              se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = join_with_no_k,
              aes(color = "Married filing jointly without a qualifying child",
                  linetype= "Married filing jointly without a qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_one_k,
              aes(color = "Married filing jointly with one qualifying child",
                  linetype= "Married filing jointly with one qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_two_k,
              aes(color = "Married filing jointly with two qualifying children",
                  linetype="Married filing jointly with two qualifying children"),
              se = FALSE,size = 1.25) +
  scale_x_continuous(limits=c(0,250000),labels=c("0","50K","100K","150K","200K","250K"))+
  scale_y_continuous(breaks=c(0,500, 1000, 1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000, 5500), limits=c(0,5800), 
                     labels=c("$0","$500","$1,000","$1,500","$2,000","$2,500","$3,000","$3,500","$4,000","$4,500","$5,000","$5,500"))+
  scale_color_manual(name = "Function", values = c("Single without a qualifying child"="purple",
                                                   "Head of household with one qualifying child"="firebrick3",
                                                   "Married filing jointly without a qualifying child"="steelblue",
                                                   "Married filing jointly with one qualifying child"="darkorange1",
                                                   "Married filing jointly with two qualifying children"="darkseagreen")) +
  scale_linetype_manual(name = "Function",values = c("Single without a qualifying child" = "solid", 
                                                     "Head of household with one qualifying child" = "dotted", 
                                                     "Married filing jointly without a qualifying child" = "dotdash",
                                                     "Married filing jointly with one qualifying child" = "dashed",
                                                     "Married filing jointly with two qualifying children" = "twodash"))+
  theme_classic()+
  theme(
    # Remove panel border
    panel.border = element_blank(),  
    # Remove panel grid lines
    panel.grid.major = element_blank(),
    # Remove panel background
    panel.background = element_blank())+
  theme(legend.position="none")+
  labs(x="Adjusted Gross Income", y="Economic Impact Payment")+
  theme(text = element_text(family="serif"),
        axis.text.x = element_text(size = 20),
        axis.text.y = element_text(size = 20),
        axis.title = element_text(size = 20),
        legend.text = element_text(size = 20))+
  geom_vline(xintercept = 150000,size = 0.2, linetype = "dashed") +
  geom_vline(xintercept = 75000,size = 0.2, linetype = "dashed") + 
  geom_vline(xintercept = 112500,size = 0.2, linetype = "dashed")

ggsave("Figure I (a) EIP I.pdf",height= 10, width= 15)

# Second EIP 

# Single with no qualifying child
sin_with_no_k <- function(x) ifelse(0 <= x & x <= 75000, 600,
                                    ifelse(x >= 75000 & x <= 87000,  600-0.05*(x-75000),
                                           ifelse(x >= 87000, 0, 600)))

# Head of household with one qualifying child
hoh_with_one_k <- function(x) ifelse(0 <= x & x <= 112500, 1200,
                                     ifelse(x >= 112500 & x <= 13600, 1200-0.05*(x-112500),
                                            ifelse(x >= 136000, 0, 1200)))

# Married Filing Jointly with no qualifying child
join_with_no_k <- function(x) ifelse(0 <= x & x <= 150000, 1200,
                                     ifelse(x >= 150000 & x <= 174000,  1200-0.05*(x-150000),
                                            ifelse(x >= 174000, 0, 1200)))

# Married Filing Jointly with one qualifying child
join_with_one_k <- function(x) ifelse(0 <= x & x <= 150000, 1800,
                                      ifelse(x >= 150000 & x <= 186000,  1800-0.05*(x-150000),
                                             ifelse(x >= 186000, 0, 1800)))

# Married Filing Jointly with two qualifying child
join_with_two_k <- function(x) ifelse(0 <= x & x <= 150000, 2400,
                                      ifelse(x >= 150000 & x <= 198000,  2400-0.05*(x-150000),
                                             ifelse(x >= 198000, 0, 2400)))

p + geom_smooth(stat="function",
                fun = sin_with_no_k,
                aes(color = "Single without a qualifying child" ,
                    linetype = "Single without a qualifying child"),
                se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = hoh_with_one_k,
              aes(color = "Head of household with one qualifying child",
                  linetype= "Head of household with one qualifying child"),
              se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = join_with_no_k,
              aes(color = "Married filing jointly without a qualifying child",
                  linetype= "Married filing jointly without a qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_one_k,
              aes(color = "Married filing jointly with one qualifying child",
                  linetype= "Married filing jointly with one qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_two_k,
              aes(color = "Married filing jointly with two qualifying children",
                  linetype="Married filing jointly with two qualifying children"),
              se = FALSE,size = 1.25) +
  scale_x_continuous(limits=c(0,250000),labels=c("0","50K","100K","150K","200K","250K"))+
  scale_y_continuous(breaks=c(0,500, 1000, 1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000, 5500), limits=c(0,5800), 
                     labels=c("$0","$500","$1,000","$1,500","$2,000","$2,500","$3,000","$3,500","$4,000","$4,500","$5,000","$5,500"))+
  scale_color_manual(name = "Function", values = c("Single without a qualifying child"="purple",
                                                   "Head of household with one qualifying child"="firebrick3",
                                                   "Married filing jointly without a qualifying child"="steelblue",
                                                   "Married filing jointly with one qualifying child"="darkorange1",
                                                   "Married filing jointly with two qualifying children"="darkseagreen")) +
  scale_linetype_manual(name = "Function",values = c("Single without a qualifying child" = "solid", 
                                                     "Head of household with one qualifying child" = "dotted", 
                                                     "Married filing jointly without a qualifying child" = "dotdash",
                                                     "Married filing jointly with one qualifying child" = "dashed",
                                                     "Married filing jointly with two qualifying children" = "twodash"))+
  theme_classic()+
  theme(
    # Remove panel border
    panel.border = element_blank(),  
    # Remove panel grid lines
    panel.grid.major = element_blank(),
    # Remove panel background
    panel.background = element_blank())+
  theme(legend.position="none")+
  labs(x="Adjusted Gross Income", y="Economic Impact Payment")+
  theme(text = element_text(family="serif"),
        axis.text.x = element_text(size = 20),
        axis.text.y = element_text(size = 20),
        axis.title = element_text(size = 20),
        legend.text = element_text(size = 20))+
  geom_vline(xintercept = 150000,size = 0.2, linetype = "dashed") +
  geom_vline(xintercept = 75000,size = 0.2, linetype = "dashed") + 
  geom_vline(xintercept = 112500,size = 0.2, linetype = "dashed")

ggsave("Figure II (b) EIP II.pdf",height= 10, width= 15)

# Third EIP 

# Single with no qualifying child
sin_with_no_k <- function(x) ifelse(0 <= x & x <= 75000, 1400,
                                    ifelse(x >= 75000 & x <= 80000,  1400-(1400/(80000-75000))*(x-75000),
                                           ifelse(x >= 80000, 0, 1400)))

# Head of household with one qualifying child
hoh_with_one_k <- function(x) ifelse(0 <= x & x <= 112500, 2800,
                                     ifelse(x >= 112500 & x <= 12000, 2800-(2800/(120000-112500))*(x-112500),
                                            ifelse(x >= 120000, 0, 2800)))

# Married Filing Jointly with no qualifying child
join_with_no_k <- function(x) ifelse(0 <= x & x <= 150000, 2800,
                                     ifelse(x >= 150000 & x <= 180000,  2800-(2800/(180000-150000))*(x-150000),
                                            ifelse(x >= 180000, 0, 2800)))

# Married Filing Jointly with one qualifying child
join_with_one_k <- function(x) ifelse(0 <= x & x <= 150000, 4200,
                                      ifelse(x >= 150000 & x <= 180000,  4200-(4200/(180000-150000))*(x-150000),
                                             ifelse(x >= 180000, 0, 4200)))

# Married Filing Jointly with two qualifying child
join_with_two_k <- function(x) ifelse(0 <= x & x <= 150000, 5600,
                                      ifelse(x >= 150000 & x <= 180000,  5600-(5600/(180000-150000))*(x-150000),
                                             ifelse(x >= 180000, 0, 5600)))

p + geom_smooth(stat="function",
                fun = sin_with_no_k,
                aes(color = "Single without a qualifying child" ,
                    linetype = "Single without a qualifying child"),
                se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = hoh_with_one_k,
              aes(color = "Head of household with one qualifying child",
                  linetype= "Head of household with one qualifying child"),
              se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = join_with_no_k,
              aes(color = "Married filing jointly without a qualifying child",
                  linetype= "Married filing jointly without a qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_one_k,
              aes(color = "Married filing jointly with one qualifying child",
                  linetype= "Married filing jointly with one qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_two_k,
              aes(color = "Married filing jointly with two qualifying children",
                  linetype="Married filing jointly with two qualifying children"),
              se = FALSE,size = 1.25) +
  scale_x_continuous(limits=c(0,250000),labels=c("0","50K","100K","150K","200K","250K"))+
  scale_y_continuous(breaks=c(0,500, 1000, 1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000, 5500), limits=c(0,5800), 
                     labels=c("$0","$500","$1,000","$1,500","$2,000","$2,500","$3,000","$3,500","$4,000","$4,500","$5,000","$5,500"))+
  scale_color_manual(name = "Function", values = c("Single without a qualifying child"="purple",
                                                   "Head of household with one qualifying child"="firebrick3",
                                                   "Married filing jointly without a qualifying child"="steelblue",
                                                   "Married filing jointly with one qualifying child"="darkorange1",
                                                   "Married filing jointly with two qualifying children"="darkseagreen")) +
  scale_linetype_manual(name = "Function",values = c("Single without a qualifying child" = "solid", 
                                                     "Head of household with one qualifying child" = "dotted", 
                                                     "Married filing jointly without a qualifying child" = "dotdash",
                                                     "Married filing jointly with one qualifying child" = "dashed",
                                                     "Married filing jointly with two qualifying children" = "twodash"))+
  theme_classic()+
  theme(
    # Remove panel border
    panel.border = element_blank(),  
    # Remove panel grid lines
    panel.grid.major = element_blank(),
    # Remove panel background
    panel.background = element_blank())+
  theme(legend.position="none")+
  labs(x="Adjusted Gross Income", y="Economic Impact Payment")+
  theme(text = element_text(family="serif"),
        axis.text.x = element_text(size = 20),
        axis.text.y = element_text(size = 20),
        axis.title = element_text(size = 20),
        legend.text = element_text(size = 20))+
  geom_vline(xintercept = 150000,size = 0.2, linetype = "dashed") +
  geom_vline(xintercept = 75000,size = 0.2, linetype = "dashed") + 
  geom_vline(xintercept = 112500,size = 0.2, linetype = "dashed")

ggsave("Figure II (c) EIP III.pdf",height= 10, width= 15)

leg <- p + geom_smooth(stat="function",
                       fun = sin_with_no_k,
                       aes(color = "Single without a qualifying child" ,
                           linetype = "Single without a qualifying child"),
                       se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = hoh_with_one_k,
              aes(color = "Head of household with one qualifying child",
                  linetype= "Head of household with one qualifying child"),
              se = FALSE,size = 1.25)+
  geom_smooth(stat="function",
              fun = join_with_no_k,
              aes(color = "Married filing jointly without a qualifying child",
                  linetype= "Married filing jointly without a qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_one_k,
              aes(color = "Married filing jointly with one qualifying child",
                  linetype= "Married filing jointly with one qualifying child"),
              se = FALSE,size = 1.25) +
  geom_smooth(stat="function",
              fun = join_with_two_k,
              aes(color = "Married filing jointly with two qualifying children",
                  linetype="Married filing jointly with two qualifying children"),
              se = FALSE,size = 1.25) +
  scale_x_continuous(limits=c(0,250000),labels=c("0","50K","100K","150K","200K","250K"))+
  scale_y_continuous(breaks=c(0,500, 1000, 1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000, 5500), limits=c(0,5800),
                     labels=c("$0","$500","$1,000","$1,500","$2,000","$2,500","$3,000","$3,500","$4,000","$4,500","$5,000","$5,500"))+
  scale_color_manual(name = "Function", values = c("Single without a qualifying child"="purple",
                                                   "Head of household with one qualifying child"="firebrick3",
                                                   "Married filing jointly without a qualifying child"="steelblue",
                                                   "Married filing jointly with one qualifying child"="darkorange1",
                                                   "Married filing jointly with two qualifying children"="darkseagreen")) +
  scale_linetype_manual(name = "Function",values = c("Single without a qualifying child" = "solid", 
                                                     "Head of household with one qualifying child" = "dotted", 
                                                     "Married filing jointly without a qualifying child" = "dotdash",
                                                     "Married filing jointly with one qualifying child" = "dashed",
                                                     "Married filing jointly with two qualifying children" = "twodash"))+
  theme_classic()+
  theme(
    # Remove panel border
    panel.border = element_blank(),  
    # Remove panel grid lines
    panel.grid.major = element_blank(),
    # Remove panel background
    panel.background = element_blank())+
  theme(legend.position = "bottom",
        legend.title = element_blank(),
        legend.spacing.y = unit(0, "mm"), 
        aspect.ratio = 1, axis.text = element_text(colour = 1, size = 12),
        legend.background = element_blank(),
        legend.key.size = unit(1.4, "cm"),
        legend.box.background = element_rect(colour = "black"))+
  guides(color = guide_legend(nrow = 3, byrow = TRUE))+
  labs(x="Adjusted Gross Income", y="Economic Impact Payment")+
  theme(text = element_text(family="serif"),
        axis.text.x = element_text(size = 20),
        axis.text.y = element_text(size = 20),
        axis.title = element_text(size = 20),
        legend.text = element_text(size = 20))+
  geom_vline(xintercept = 150000,size = 0.2, linetype = "dashed") +
  geom_vline(xintercept = 75000,size = 0.2, linetype = "dashed") + 
  geom_vline(xintercept = 112500,size = 0.2, linetype = "dashed")

extract_legend <- function(my_ggp) {
  step1 <- ggplot_gtable(ggplot_build(my_ggp))
  step2 <- which(sapply(step1$grobs, function(x) x$name) == "guide-box")
  step3 <- step1$grobs[[step2]]
  return(step3)
}

shared_legend <- extract_legend(leg) 

grid.arrange(arrangeGrob(shared_legend, ncol = 1))

ggsave("Figure II Legend.pdf",height= 10, width= 15)


### Figure III ####

# Data Source : EIP Dashboard (Bureau of the Fiscal Service)

# First EIP 
type <- rep(c("Direct Deposit","Check","Debit Card"),9)

time <- c("Apr","Apr","Apr","May","May","May","Jun","Jun","Jun",
          "Jul","Jul","Jul","Aug","Aug","Aug","Sep","Sep","Sep",
          "Oct","Oct","Oct","Nov","Nov","Nov","Dec","Dec","Dec")

value <- c(104.7,7.0,0,
           15.1,26.8,3.7,
           0.7,1.7,0,
           0.6,1.0,0,
           0.6,0.7,0,
           0.5,0.3,0,
           0.4,0.8,0,
           0.6,0.7,0,
           0.1,0.2,0)

amt_disb_by_wk <- data.frame(type,time,value)

amt_disb_by_wk$type <- factor(amt_disb_by_wk$type , levels = c("Debit Card","Check","Direct Deposit"))

amt_disb_by_wk$time <- factor(amt_disb_by_wk$time , levels = c("Apr", "May", "Jun", "Jul",
                                                               "Aug", "Sep", "Oct",
                                                               "Nov", "Dec"))

p2b <- ggplot(amt_disb_by_wk, aes(x = time,y = value, fill = type)) +
  geom_col(width = 0.7)+
  scale_fill_manual(
    values = c("Direct Deposit" = "steelblue4",
               "Check"="khaki",
               "Debit Card" = "darkolivegreen3"))+
  theme(
    # Remove panel border
    #panel.border = element_blank(),  
    # Remove panel grid lines
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    # Remove panel background
    panel.background = element_blank(),
    # Remove axis line
    axis.line = element_blank(),
    # Remove legend title
    legend.title = element_blank(),
    #Remove x ticks
    axis.ticks = element_blank(),
    # Modify y ticks
    # Change y-axis angle
    #axis.title.y = element_text(angle=0,vjust=1.1,hjust=3),
    # Chane legend position
    legend.position = "top")+
  #Change label text
  theme(text = element_text(face="bold"),
        axis.text.x = element_text(size = 40, angle = 75, vjust = 0.6),
        axis.text.y = element_text(size = 40),
        axis.title.y = element_text( size=40, face="plain"),
        axis.title = element_text(size = 40),
        legend.text = element_text(size = 40))+
  # change y scale step size
  scale_y_continuous(breaks = seq(0, 130, 10))+
  labs(x=NULL,y="Number of payments (in millions)") 

p2b 
ggsave("Figure III(a).pdf",height= 13, width= 14)

#Third EIP 
type <- rep(c("Direct Deposit","Check","Debit Card"),10)

time <- c("Mar","Mar","Mar","Apr","Apr","Apr","May","May","May","Jun","Jun","Jun",
          "Jul","Jul","Jul","Aug","Aug","Aug","Sep","Sep","Sep",
          "Oct","Oct","Oct","Nov","Nov","Nov","Dec","Dec","Dec")

value <- c(109.7,15.1,4.8,
           26.7,6.3,0,
           2.0,2.0,0,
           1.9,1.5,0,
           0.7,0.8,0,
           0.7,0.6,0,
           0.7,0.5,0,
           0.3,0.5,0,
           0.4,0.4,0,
           0.1,0.1,0)

amt_disb_by_wk <- data.frame(type,time,value)

amt_disb_by_wk$type <- factor(amt_disb_by_wk$type , levels = c("Debit Card","Check","Direct Deposit"))

amt_disb_by_wk$time <- factor(amt_disb_by_wk$time , levels = c("Mar","Apr", "May", "Jun", "Jul",
                                                               "Aug", "Sep", "Oct",
                                                               "Nov", "Dec"))

p2b <- ggplot(amt_disb_by_wk, aes(x = time,y = value, fill = type)) +
  geom_col(width = 0.7)+
  scale_fill_manual(
    values = c("Direct Deposit" = "steelblue4",
               "Check"="khaki",
               "Debit Card" = "darkolivegreen3"))+
  theme(
    # Remove panel border
    #panel.border = element_blank(),  
    # Remove panel grid lines
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    # Remove panel background
    panel.background = element_blank(),
    # Remove axis line
    axis.line = element_blank(),
    # Remove legend title
    legend.title = element_blank(),
    #Remove x ticks
    axis.ticks = element_blank(),
    # Modify y ticks
    # Change y-axis angle
    #axis.title.y = element_text(angle=0,vjust=1.1,hjust=3),
    # Chane legend position
    legend.position = "top")+
  #Change label text
  theme(text = element_text(face="bold"),
        axis.text.x = element_text(size = 40, angle = 75, vjust = 0.6),
        axis.text.y = element_text(size = 40),
        axis.title.y = element_text( size=40, face="plain"),
        axis.title = element_text(size = 40),
        legend.text = element_text(size = 40))+
  # change y scale step size
  scale_y_continuous(breaks = seq(0, 130, 10))+
  labs(x=NULL,y="Number of payments (in millions)") 

p2b 
ggsave("Figure III(b).pdf",height= 13, width= 14)

